## Gifting rates - what % of other assets given as gifts?

# Prelims -----------------------------------------------------------------

rm(list=ls())
gc()


source("./R scripts/Data and parameters/0 Custom functions.R")


# Read HILDA grouped master -----------------------------------------------

hilda_grouped <- qread("./Input data/Intermediate input data/hilda_grouped_master.qs")



# Gift giving rates -------------------------------------------------------------

## using data on gifts received, what proportion of parents by age are giving a gift? and what is the amount they are giving?
## (approximate because parents can have many kids but also kids can have many parents)
gift_by_donor_age <- hilda_grouped %>% 
  ## responding persons who stated they got a gift or got 0 gifts
  filter(gifts>=0 & hhwtrp>=0) %>% ## note this excludes not stated and don't know responses among responding persons, based on how the variable is constructed
  ## keep only ppl who have parent age recorded...
  filter(!is.na(parent_age)) %>% 
  mutate(gift_giver= ifelse(gifts>0, 1, 0) ) %>% ## gift giver from parent perspective

  group_by(wavenumber, parent_age_grp) %>% ## by wave and age
  summarise(gift_giver_num = sum(gift_giver*hhwtrp), ## number of parents who gave a gift (roughly)
            gift_giver_prop = sum(gift_giver*hhwtrp)/sum(hhwtrp), ## proportion of parents who give gifts
            gift_size_tot = sum(gifts*hhwtrp), ## total gifts given 
            gift_size_av = sum(gifts*hhwtrp)/sum(gift_giver*hhwtrp) ## average size of gift given, among those giving a gift
  ) 

## need to use above info on parents who gave gifts, along with how manyppl in hilda are parents 
## below calcs:
## (1) estimate of percentage of ppl in whole age group who give gifts = proportion of each age group with children * gift_giver_prop 
## (2) estimate of number of ppl who give gifts = (1) * number of people
## (3) estimate of total gifts given = average gift given * (2)
## (4) estimate of percentage of other assets that ppl in whole age group give to their kids = (3) / total other assets
## note, wealth only available for waves 2,6,10,14,18

hilda_parents <- hilda_grouped %>% 
  ## filter to responding persons (asked q's on children, gifts/bequests) in wealth waves
  filter(wavenumber %in% c(2,6,10,14,18) & hhwtrp>=0) %>% 
  ## determine whether person has children (resident and non resident)
  mutate(has_child = ifelse(tcr>=1 | tcnr>=1, 1, 0)) %>%
  group_by(wavenumber, age_grp) %>% 
  summarise(has_child_prop = sum(has_child*hhwtrp)/sum(hhwtrp), ## what proportion of each age group has children?
            other_assets = sum(other_assets*hhwtrp), ## total other assets
            n = sum(hhwtrp)
  ) %>% 
  ## merge with above data on gift givers among parents
  left_join(gift_by_donor_age, by=c("wavenumber", "age_grp"="parent_age_grp")) %>% 
  ## fill missing values with 0
  mutate(across(matches("gift_giver|gift_size"), ~ifelse(is.na(.x), 0, .x))) %>% 
  ## calculate 
  mutate(gift_giver_prop_all = gift_giver_prop*has_child_prop, ## (1) proportion of all people who give gifts
         gift_giving_rate = (gift_size_av*gift_giver_prop_all*n)/other_assets ## (4) proportion of other assets given as gifts
  )

## all age groups
gift_rates <-  hilda_parents %>% 
  ## average gift giving rate across all waves
  group_by(age_grp) %>% 
  summarise(gift_giving_rate = mean(gift_giving_rate)) %>% 
  ## add missing age groups
  right_join(hilda_grouped %>% distinct(age_grp)) %>% 
  mutate(gift_giving_rate = ifelse(is.na(gift_giving_rate), 0, gift_giving_rate)) %>% 
  ## assume no gifts given when you are 80+
  mutate(gift_giving_rate = ifelse(age_grp>="[80,85)", 0, gift_giving_rate)) %>% 
  arrange(age_grp) %>% 
  ##smooth
  mutate(gift_giving_rate = custom_smoother(gift_giving_rate))


## plot averaged across waves
# ggplot(gift_rates) +
#   geom_col(aes(x=age_grp, y=gift_giving_rate)) 


## benchmark the rates such that when applied to 2018 other assets, they will produce total gifts given comparable to total gifts received in 2018
hilda_grouped_gift <- hilda_grouped %>% 
  select(wavenumber, hhwtrp, age_grp, gifts, other_assets) %>% 
  filter(wavenumber==18) %>% 
  left_join(gift_rates) %>% 
  mutate(total_gifts_received_row = ifelse(gifts>=0 & hhwtrp>=0, hhwtrp*gifts, NA),
         total_gifts_received = sum(total_gifts_received_row, na.rm=T),
         total_gifts_given = sum(other_assets*gift_giving_rate*hhwtrp),
         scale_factor = total_gifts_received / total_gifts_given,
         total_gifts_given_scaled = sum(other_assets*gift_giving_rate*scale_factor*hhwtrp) )


## apply scale factor to gifting rates
gift_rates_scaled <- gift_rates %>% 
  mutate(gift_giving_rate = gift_giving_rate*hilda_grouped_gift[[1, "scale_factor"]])

## USED IN APPENDIX
## The 50–54 age group had the largest rate, gifting 0.9 per cent of their other assets each year.
gift_rates_scaled %>% filter(gift_giving_rate==max(gift_giving_rate))

# ggplot(gift_rates_scaled) +
#   geom_col(aes(x=age_grp, y=gift_giving_rate)) 


## save
qsave(gift_rates_scaled, "./Input data/gift_giving_rate_a.qs")
